/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2026 i-net software GmbH, Berlin, Germany.
**/
package rdc;


import java.util.ArrayList;
import java.util.List;

import com.inet.report.DatabaseField;
import com.inet.report.DatabaseTables;
import com.inet.report.Datasource;
import com.inet.report.Engine;
import com.inet.report.Field;
import com.inet.report.Fields;
import com.inet.report.RDC;
import com.inet.report.TableSource;

/**
 * This sample shows how to create a mailing label report with some database fields. The Data Source Configuration
 * "Sample Database" is necessary for this sample. You can create it using the Data Source Manager in i-net Designer or
 * Configuration Manager.
 */

public class MailingLabelSample extends RDCSample {

    static final String[] ORDERS_NAMES = { "OrderID", "OrderAmount", "CustomerID", "EmployeeID", "OrderDate", "RequiredDate", "ShipDate", "ShipVia",
        "Shipped", "PO#", "PaymentReceived" };
    private int[]         ordersTypes  = { Field.NUMBER, Field.NUMBER, Field.STRING, Field.NUMBER, Field.DATETIME, Field.DATETIME, Field.DATETIME,
        Field.STRING, Field.BOOLEAN, Field.STRING, Field.BOOLEAN };

    /**
     * Creates a new engine, adds some database fields to it and create a mailing label.
     * @param exportFmt the output format (e.g. Engine.EXPORT_PDF)
     * @return the new engine for the report
     */
    @Override
    public Engine createAndFillEngine( String exportFmt ) {

        try {
            // Create an Engine for a new report
            Engine eng = RDC.createEmptyEngine( exportFmt );

            // Fill the engine
            final DatabaseTables dbTables = eng.getDatabaseTables();
            final Fields fields = eng.getFields();

            // Define a data source
            Datasource ds = dbTables.createDatasource( "Sample Database" ); // Data Source Configuration "Sample Database"

            // Define the table of the data source
            TableSource ts_Orders = ds.createTableSource( "Orders" );
            for( int colIdx = 0; colIdx < ORDERS_NAMES.length; colIdx++ ) {
                ts_Orders.addColumn( ORDERS_NAMES[colIdx], ordersTypes[colIdx] );
            }

            // Add some database fields to the report
            DatabaseField a = fields.getDatabaseField( ts_Orders.getAlias() + ".OrderID" );
            DatabaseField b = fields.getDatabaseField( ts_Orders.getAlias() + ".Shipped" );
            DatabaseField c = fields.getDatabaseField( ts_Orders.getAlias() + ".CustomerID" );
            List<Field> fieldsV = new ArrayList<>();
            fieldsV.add( a );
            fieldsV.add( b );
            fieldsV.add( c );

            /*
             * This method creates a report for mailing labels.
             * You have to specify the height and width for one label and the distance between labels (horizontal and vertical). 
             * The page will be filled with labels as much as it is possible until (across and down).
             * The printing direction has two states: across than down, a row will be filled first and down than across, a
             * column will be filled first.
             */
            eng = RDC.createMailingLabel( eng, 2000, 3000, 100, 100, false, fieldsV );

            return eng;
        } catch( Throwable e ) {
            e.printStackTrace();
            System.exit( 0 );
            return null;
        }
    }

    /**
     * Main method of this sample
     * @param args arguments not used
     */
    public static void main( String[] args ) {
        new MailingLabelSample().initUI();
    }
}
